function thrust(P,V)
% P given in kPa
% V given in mL

%%%%%%%%%% initial conditions
global Vtotal Vwo Patm Pao Aa Ae rho;

Vtotal = 0.000340;  % total chamber volume (m^3)
Vwo=V/1000000;      % convert to m^3

Patm = 100000;      % atmospheric pressure (Pa)
Pao = P*1000;       % convert to Pa

da = 0.0381;        % chamber diameter (m)
de = 0.0124;        % nozzle exit diameter (m)
Aa = pi*(da/2)^2; % chamber cross-section area
Ae = pi*(de/2)^2; % nozzle exit cross-section area

rho = 1000; % water density, kg/m^3

%%%%%%%%% Run the simulation

tstop = 0.2;

%Run ode45 on Va
[t,Va] = ode45(@state_eq,[0 tstop],(Vtotal-Vwo));   

% note that everything below must be done using vectors returned from ode45

% calculate the air pressure using the equation given under "Air Expansion"
Pa = (Pao * (Va/(Vtotal-Vwo)).^(-1.4)); 

% if Va has gone to zero, we're out of water.  However, there may still be
% positive pressure in the chamber.  Without modeling the compressible flow
% phase for the air exiting the chamber, we can just treat this as a
% discontinuity and assume the force from the air is negligible
Pa = Patm + (Va <= Vtotal).*(Pa-Patm);

% calculate the exit velocity using Bernoulli's equation given under 
% "Water Nozzle Exit Velocity", with va removed in the same manner as 
% described in state_eq.m
ve = sqrt(2*(Pa - Patm)./(rho * (1 - (Ae/Aa)^2)));

% calculate the thrust force using the equation given under "Thrust"
F = Ae * rho * ve.^2; % thrust

% plot the results, showing the data points
plot(t,F,'r.-');
xlabel('Time (sec)');
ylabel('Thrust (N)');
title(['Jonathan Fiene, P=' num2str(P) 'kPa, V=' num2str(V) 'mL']);
grid on;

